const express = require('express');
const http = require('http');
const socketIo = require('socket.io');
const cors = require('cors');
const jwt = require('jsonwebtoken');
const bcrypt = require('bcrypt');
const multer = require('multer');
const sharp = require('sharp');
const path = require('path');
const fs = require('fs');
const db = require('./database');
const { sendNotificationEmail } = require('./emailService');
const { encrypt, decrypt } = require('./encryption');
require('dotenv').config();

const app = express();
const server = http.createServer(app);
const io = socketIo(server, {
  cors: {
    origin: '*',
    credentials: true
  },
  maxHttpBufferSize: 10e6 // 10MB for file uploads
});

// Serve static files first
app.use(express.static('public'));
app.use('/uploads', express.static('uploads'));

app.use(cors({
  origin: '*',
  credentials: true
}));
app.use(express.json());

// Multer setup for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, 'uploads/');
  },
  filename: (req, file, cb) => {
    const uniqueName = `${Date.now()}-${Math.round(Math.random() * 1E9)}${path.extname(file.originalname)}`;
    cb(null, uniqueName);
  }
});

const upload = multer({
  storage,
  limits: { fileSize: 10 * 1024 * 1024 }, // 10MB limit
  fileFilter: (req, file, cb) => {
    const allowedTypes = /jpeg|jpg|png|gif|webp|mp3|wav|ogg|webm/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);
    
    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('Invalid file type'));
    }
  }
});

// Authentication endpoint
app.post('/api/auth', async (req, res) => {
  const { code } = req.body;
  
  db.get('SELECT * FROM access_codes WHERE code = ?', [code], async (err, row) => {
    if (err || !row) {
      // Get client information for failed login alert
      const clientIp = req.headers['x-forwarded-for'] || req.connection.remoteAddress;
      const userAgent = req.headers['user-agent'] || 'Unknown';
      const attemptTime = new Date().toLocaleString();
      const referrer = req.headers['referer'] || 'Direct';
      
      // Send alert email for failed login
      try {
        const { sendEmail } = require('./emailService');
        await sendEmail(
          'wolfganggermain1@gmail.com',
          'Security Alert: Failed Login Attempt',
          `
          <h2>Failed Login Attempt Detected</h2>
          <p><strong>Time:</strong> ${attemptTime}</p>
          <p><strong>Attempted Code:</strong> ${code}</p>
          <p><strong>IP Address:</strong> ${clientIp}</p>
          <p><strong>User Agent:</strong> ${userAgent}</p>
          <p><strong>Referrer:</strong> ${referrer}</p>
          <p><strong>Request Headers:</strong></p>
          <pre>${JSON.stringify(req.headers, null, 2)}</pre>
          <hr>
          <p style="color: red;"><strong>Action Required:</strong> Someone attempted to access the secure chat with an invalid code.</p>
          `
        );
        console.log('Failed login alert sent to wolfganggermain1@gmail.com');
      } catch (emailError) {
        console.error('Failed to send security alert:', emailError);
      }
      
      return res.status(401).json({ error: 'Invalid access code' });
    }
    
    const token = jwt.sign(
      { code, roomId: row.room_id },
      process.env.JWT_SECRET,
      { expiresIn: '24h' }
    );
    
    // Update user as online
    db.run('UPDATE user_sessions SET is_online = 1, last_seen = CURRENT_TIMESTAMP WHERE code = ?', [code]);
    
    // Get chat history
    db.all(
      'SELECT * FROM messages WHERE room_id = ? ORDER BY timestamp DESC LIMIT 50',
      [row.room_id],
      (err, messages) => {
        res.json({
          token,
          roomId: row.room_id,
          messages: messages ? messages.reverse() : []
        });
      }
    );
  });
});

// Delete messages endpoint
app.delete('/api/messages', async (req, res) => {
  try {
    const authHeader = req.headers.authorization;
    if (!authHeader) {
      return res.status(401).json({ error: 'No authorization header' });
    }

    const token = authHeader.split(' ')[1];
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    
    // Delete all messages for the user's room
    db.run(
      'DELETE FROM messages WHERE room_id = ?',
      [decoded.roomId],
      (err) => {
        if (err) {
          return res.status(500).json({ error: 'Failed to delete messages' });
        }
        res.json({ success: true, message: 'All messages deleted' });
      }
    );
  } catch (error) {
    console.error('Delete messages error:', error);
    res.status(500).json({ error: 'Failed to delete messages' });
  }
});

// File upload endpoint
app.post('/api/upload', upload.single('file'), async (req, res) => {
  try {
    const authHeader = req.headers.authorization;
    if (!authHeader) {
      return res.status(401).json({ error: 'No authorization header' });
    }

    const token = authHeader.split(' ')[1];
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    
    if (!req.file) {
      return res.status(400).json({ error: 'No file uploaded' });
    }

    let processedFilePath = req.file.path;

    // Process images - compress and resize
    if (req.file.mimetype.startsWith('image/')) {
      const processedName = `processed-${req.file.filename}`;
      processedFilePath = path.join('uploads', processedName);
      
      await sharp(req.file.path)
        .resize(1200, 1200, { 
          fit: 'inside',
          withoutEnlargement: true 
        })
        .jpeg({ quality: 80 })
        .toFile(processedFilePath);
      
      // Delete original
      fs.unlinkSync(req.file.path);
    }

    res.json({
      success: true,
      filePath: processedFilePath,
      fileType: req.file.mimetype,
      originalName: req.file.originalname
    });
  } catch (error) {
    console.error('Upload error:', error);
    res.status(500).json({ error: 'Upload failed' });
  }
});

// Socket.io connection handling
const activeUsers = new Map();

io.use((socket, next) => {
  const token = socket.handshake.auth.token;
  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    socket.userId = decoded.code;
    socket.roomId = decoded.roomId;
    next();
  } catch (err) {
    next(new Error('Authentication error'));
  }
});

io.on('connection', (socket) => {
  console.log('User connected:', socket.userId);
  
  // Join room
  socket.join(socket.roomId);
  activeUsers.set(socket.userId, socket.id);
  
  // Update online status
  db.run('UPDATE user_sessions SET is_online = 1, last_seen = CURRENT_TIMESTAMP WHERE code = ?', [socket.userId]);
  
  // Notify room about online users
  const roomUsers = Array.from(activeUsers.entries())
    .filter(([userId]) => {
      const userSocket = io.sockets.sockets.get(activeUsers.get(userId));
      return userSocket && userSocket.roomId === socket.roomId;
    })
    .map(([userId]) => userId);
  
  io.to(socket.roomId).emit('users_online', roomUsers);
  
  // Handle messages (text, image, voice)
  socket.on('send_message', (messageData) => {
    const fullMessage = {
      id: Date.now(),
      sender: socket.userId,
      message_type: messageData.type || 'text',
      encrypted_message: JSON.stringify(messageData.encryptedContent),
      file_path: messageData.filePath || null,
      timestamp: new Date().toISOString()
    };
    
    // Save to database
    db.run(
      'INSERT INTO messages (room_id, sender_code, message_type, encrypted_message, file_path) VALUES (?, ?, ?, ?, ?)',
      [socket.roomId, socket.userId, fullMessage.message_type, fullMessage.encrypted_message, fullMessage.file_path]
    );
    
    // Broadcast to room
    io.to(socket.roomId).emit('new_message', fullMessage);
    
    // Check if other user is offline and send email
    console.log('Current room users:', roomUsers);
    console.log('Active users:', Array.from(activeUsers.keys()));
    
    // Get all users in the room from database
    db.all('SELECT DISTINCT code FROM user_sessions WHERE code IN (SELECT code FROM access_codes WHERE room_id = ?)', 
      [socket.roomId], 
      (err, allRoomUsers) => {
        if (err) {
          console.error('Error getting room users:', err);
          return;
        }
        
        allRoomUsers.forEach(({ code }) => {
          if (code !== socket.userId && !activeUsers.has(code)) {
            console.log(`User ${code} is offline, sending email...`);
            db.get('SELECT email FROM user_sessions WHERE code = ?', [code], (err, row) => {
              if (err) {
                console.error('Error getting email:', err);
              } else if (row && row.email) {
                console.log(`Sending email to ${row.email} for offline user ${code}`);
                sendNotificationEmail(row.email, socket.userId);
              }
            });
          }
        });
      }
    );
  });
  
  // Handle typing
  socket.on('typing', (isTyping) => {
    socket.to(socket.roomId).emit('user_typing', { user: socket.userId, isTyping });
  });
  
  // Handle disconnect
  socket.on('disconnect', () => {
    console.log('User disconnected:', socket.userId);
    activeUsers.delete(socket.userId);
    
    // Update offline status
    db.run('UPDATE user_sessions SET is_online = 0 WHERE code = ?', [socket.userId]);
    
    // Notify room
    const roomUsers = Array.from(activeUsers.entries())
      .filter(([userId]) => {
        const userSocket = io.sockets.sockets.get(activeUsers.get(userId));
        return userSocket && userSocket.roomId === socket.roomId;
      })
      .map(([userId]) => userId);
    
    io.to(socket.roomId).emit('users_online', roomUsers);
  });
});

const PORT = process.env.PORT || 3001;
server.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});